<?php
namespace App\Http\Controllers;


use App\Models\Salon;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Controllers\DemoVideoController;
use App\Models\DemoVideo;
use App\Models\Service;
use Illuminate\Http\Request;


class SalonController extends Controller
{
    public function store(Request $request): JsonResponse
{
    $validated = $request->validate([
        'name' => 'required|string|max:255',
        'location' => 'required|string|max:255',
        'rating' => 'nullable|numeric|min:0|max:5',
        'reviews' => 'nullable|integer|min:0',
        'open_time' => 'nullable|date_format:H:i',
        'close_time' => 'nullable|date_format:H:i',
        'price_min' => 'required|numeric|min:0',
        'price_max' => 'required|numeric|min:0',
        'discount' => 'nullable|integer|min:0|max:100',
        'video_url' => 'nullable|string',
        'facilities' => 'nullable|array',
    ]);

    $salon = Salon::create($validated);

    return response()->json([
        'message' => 'Salon created successfully',
        'salon' => $salon
    ], 201);
}


    public function show($id): JsonResponse
    {
        $salon = Salon::with('services')->find($id);

        if (!$salon) {
            return response()->json(['message' => 'Salon not found'], 404);
        }

        return response()->json([
            'name' => $salon->name,
            'location' => $salon->location,
            'rating' => $salon->rating,
            'reviews' => $salon->reviews,
            'timetable' => [
                'open' => $salon->open_time,
                'close' => $salon->close_time
            ],
            'price_range' => [
                'lowest' => $salon->price_min,
                'highest' => $salon->price_max
            ],
            'video' => $salon->video_url,
            'facilities' => $salon->facilities,
            'services' => $salon->services->map(function ($service) {
                return [
                    'name' => $service->name,
                    'discount_percentage' => $service->discount_percentage,
                    'rating' => $service->rating,
                    'reviews' => $service->reviews,
                    'price' => $service->price,
                    'likes' => $service->likes
                ];
            })
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $salon = Salon::find($id);

        if (!$salon) {
            return response()->json(['message' => 'Salon not found'], 404);
        }

        $salon->delete();
        return response()->json(['message' => 'Salon deleted successfully'], 200);
    }
    public function salonDetail($salon_id): JsonResponse
    {
        // Fetch salon details
        $salon = Salon::findOrFail($salon_id);

        // Fetch demo videos for the salon
        $demoVideos = DemoVideo::where('salon_id', $salon_id)->pluck('video_url');

        // Fetch services of the salon
        $services = Service::where('salon_id', $salon_id)->get();

        return response()->json([
            'salon' => [
                'name' => $salon->name,
                'rating' => $salon->rating,
                'reviews' => $salon->reviews,
                'price' => $salon->price,
                'discount' => $salon->discount,
                'book_now' => true
            ],
            'demo_videos' => $demoVideos,
            'services' => $services
        ], 200);
    }


    public function update(Request $request, $id): JsonResponse
{
    $salon = Salon::find($id);

    if (!$salon) {
        return response()->json(['message' => 'Salon not found'], 404);
    }

    $validated = $request->validate([
        'name' => 'sometimes|required|string|max:255',
        'location' => 'sometimes|required|string|max:255',
        'rating' => 'nullable|numeric|min:0|max:5',
        'reviews' => 'nullable|integer|min:0',
        'open_time' => 'nullable|date_format:H:i',
        'close_time' => 'nullable|date_format:H:i',
        'price_min' => 'nullable|numeric|min:0',
        'price_max' => 'nullable|numeric|min:0',
        'discount' => 'nullable|integer|min:0|max:100',
        'video_url' => 'nullable|string',
        'facilities' => 'nullable|array',
    ]);

    $salon->update($validated);

    return response()->json([
        'message' => 'Salon updated successfully',
        'salon' => $salon
    ], 200);
}


}
