<?php

namespace App\Http\Controllers;

use App\Models\Qualification;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class QualificationController extends Controller
{
    // Ensure only authenticated users can access these APIs
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    // Add Qualification
    public function addQualification(Request $request): JsonResponse
    {
        $request->validate([
            'qualification_name' => 'required|string|max:255',
            'institute_name' => 'required|string|max:255',
            'year_completed' => 'required|integer|min:1900|max:' . date('Y'),
            'certificate_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $qualification = new Qualification();
        $qualification->beautician_id = $user->id;
        $qualification->qualification_name = $request->qualification_name;
        $qualification->institute_name = $request->institute_name;
        $qualification->year_completed = $request->year_completed;

        if ($request->hasFile('certificate_image')) {
            $imagePath = $request->file('certificate_image')->store('certificates', 'public');
            $qualification->certificate_image = $imagePath;
        }

        $qualification->save();

        return response()->json(['message' => 'Qualification added successfully', 'qualification' => $qualification], 201);
    }

    // View Qualifications
    public function viewQualifications(): JsonResponse
    {
        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $qualifications = Qualification::where('beautician_id', $user->id)->get();

        return response()->json(['qualifications' => $qualifications]);
    }

    // Update Qualification
    public function updateQualification(Request $request, $id): JsonResponse
    {
        $request->validate([
            'qualification_name' => 'sometimes|required|string|max:255',
            'institute_name' => 'sometimes|required|string|max:255',
            'year_completed' => 'sometimes|required|integer|min:1900|max:' . date('Y'),
            'certificate_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $qualification = Qualification::where('beautician_id', $user->id)->findOrFail($id);

        if ($request->has('qualification_name')) {
            $qualification->qualification_name = $request->qualification_name;
        }
        if ($request->has('institute_name')) {
            $qualification->institute_name = $request->institute_name;
        }
        if ($request->has('year_completed')) {
            $qualification->year_completed = $request->year_completed;
        }

        if ($request->hasFile('certificate_image')) {
            if ($qualification->certificate_image) {
                Storage::disk('public')->delete($qualification->certificate_image);
            }
            $imagePath = $request->file('certificate_image')->store('certificates', 'public');
            $qualification->certificate_image = $imagePath;
        }

        $qualification->save();

        return response()->json(['message' => 'Qualification updated successfully', 'qualification' => $qualification]);
    }

    // Delete Qualification
    public function deleteQualification($id): JsonResponse
    {
        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $qualification = Qualification::where('beautician_id', $user->id)->findOrFail($id);

        if ($qualification->certificate_image) {
            Storage::disk('public')->delete($qualification->certificate_image);
        }

        $qualification->delete();

        return response()->json(['message' => 'Qualification deleted successfully']);
    }
}
