<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PaymentOption;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Crypt;

class PaymentOptionController extends Controller
{
    public function index(Request $request)
{
    $user_id = $request->query('user_id');

    if (!$user_id) {
        return response()->json([
            'success' => false,
            'message' => 'User ID is required'
        ], 400);
    }

    $payments = PaymentOption::where('user_id', $user_id)->get();

    return response()->json([
        'success' => true,
        'data' => $payments
    ]);
}


    public function store(Request $request)
    {
        // Validate based on the selected payment method
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'payment_method' => 'required|in:credit_card,easy_paisa,jazz_cash,bank_transfer',
            'amount' => 'required|numeric|min:1',
            'currency' => 'required|string',
            'description' => 'nullable|string',

            // Credit Card
            'payment_details.card_number' => 'nullable|required_if:payment_method,credit_card|string|min:12|max:19',
            'payment_details.expiry_date' => 'nullable|required_if:payment_method,credit_card|string',
            'payment_details.cvv' => 'nullable|required_if:payment_method,credit_card|string|min:3|max:4',

            // Mobile Wallet
            'mobile_wallet.provider' => 'nullable|required_if:payment_method,easy_paisa,jazz_cash|string',
            'mobile_wallet.mobile_number' => 'nullable|required_if:payment_method,easy_paisa,jazz_cash|string|min:11|max:14',
            'mobile_wallet.transaction_id' => 'nullable|required_if:payment_method,easy_paisa,jazz_cash|string',

            // Bank Transfer
            'bank_transfer.bank_name' => 'nullable|required_if:payment_method,bank_transfer|string',
            'bank_transfer.account_number' => 'nullable|required_if:payment_method,bank_transfer|string',
            'bank_transfer.transaction_id' => 'nullable|required_if:payment_method,bank_transfer|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        // Prepare payment data
        $paymentData = [
            'user_id' => $request->user_id,
            'payment_method' => $request->payment_method,
            'amount' => $request->amount,
            'currency' => $request->currency,
            'description' => $request->description ?? null, // Handle nullable value
        ];

        // Securely store payment details based on method
        if ($request->payment_method === 'credit_card' && isset($request->payment_details)) {
            $paymentData['card_number'] = Crypt::encryptString($request->input('payment_details.card_number'));
            $paymentData['expiry_date'] = Crypt::encryptString($request->input('payment_details.expiry_date'));
            $paymentData['cvv'] = Crypt::encryptString($request->input('payment_details.cvv'));
        } elseif (in_array($request->payment_method, ['easy_paisa', 'jazz_cash']) && isset($request->mobile_wallet)) {
            $paymentData['mobile_number'] = $request->input('mobile_wallet.mobile_number');
            $paymentData['transaction_id'] = $request->input('mobile_wallet.transaction_id');
        } elseif ($request->payment_method === 'bank_transfer' && isset($request->bank_transfer)) {
            $paymentData['bank_name'] = $request->input('bank_transfer.bank_name');
            $paymentData['account_number'] = Crypt::encryptString($request->input('bank_transfer.account_number'));
            $paymentData['transaction_id'] = $request->input('bank_transfer.transaction_id');
        }

        // Create a payment option record
        $paymentOption = PaymentOption::create($paymentData);

        return response()->json([
            'success' => true,
            'message' => 'Payment option saved successfully',
            'data' => $paymentOption
        ], 201);
    }
}
