<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use App\Http\Resources\OrderResource;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller {
    public function index(Request $request) {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => 'Unauthorized access. Please log in.'
            ], Response::HTTP_UNAUTHORIZED);
        }

        $orders = [
            'to_pay' => Order::where('user_id', $user->id)->where('status', 'to_pay')->get(),
            'to_receive' => Order::where('user_id', $user->id)->where('status', 'to_receive')->get(),
            'to_review' => Order::where('user_id', $user->id)->where('status', 'to_review')->get(),
        ];

        return response()->json([
            'status' => true,
            'message' => 'Orders retrieved successfully',
            'data' => $orders
        ], Response::HTTP_OK);
    }

    public function myOrders(Request $request)
    {
        // Get authenticated user
        $user = Auth::user();

        // Fetch orders for the logged-in user
        $orders = Order::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Separate Recent and Past orders
        $recentOrders = $orders->filter(fn($order) => $order->status === 'Pending');
        $pastOrders = $orders->filter(fn($order) => in_array($order->status, ['Completed', 'Cancelled']));

        return response()->json([
            'success' => true,
            'message' => 'Orders retrieved successfully',
            'recent_orders' => OrderResource::collection($recentOrders),
            'past_orders' => OrderResource::collection($pastOrders),
            'meta' => [
                'current_page' => $orders->currentPage(),
                'per_page' => $orders->perPage(),
                'total_orders' => $orders->total(),
            ],
        ]);
    }
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'item_name' => 'required|string|max:255',
            'type' => 'required|string',
            'status' => 'required|in:to_pay,to_receive,to_review,completed,cancelled',
            'price' => 'required|numeric',
        ]);

        $order = Order::create($validated);

        return response()->json(['message' => 'Order created successfully', 'order' => $order], 201);
    }

    public function getProductSales(Request $request)
    {
        // Paginate the result to show 10 products per page
        $products = DB::table('orders')
            ->where('status', 'to_pay')
            ->select('item_name',
                     DB::raw('SUM(quantity) as total_quantity'),
                     DB::raw('SUM(price * quantity) as total_sales'))
            ->groupBy('item_name')
            ->orderByDesc('total_sales') // Optional: Sort by total sales
            ->paginate(10); // Set pagination limit here

        // Optimized sum directly from DB instead of PHP collection sum
        $totalSales = DB::table('orders')
            ->where('status', 'to_pay')
            ->select(DB::raw('SUM(price * quantity) as total'))
            ->value('total');

        return response()->json([
            'status' => true,
            'message' => 'Product Sales Overview',
            'data' => [
                'total_sales' => $totalSales ?? 0,
                'products' => $products // Paginated products
            ]
        ]);
    }



}

