<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\OrderConfirmation;
use Illuminate\Support\Str;

class OrderConfirmationController extends Controller
{
    // Generate Order Confirmation
    public function store(Request $request)
    {
        $orderId = strtoupper(Str::random(10));

        $request->validate([
            'order_date' => 'required|date',
            'expected_delivery_date' => 'required|date|after:order_date',
        ]);

        $order = OrderConfirmation::create([
            'order_id' => $orderId,
            'order_date' => $request->order_date,
            'expected_delivery_date' => $request->expected_delivery_date,
            'track_order_url' => url('/api/track-order/' . $orderId),
            'status' => 'Pending' // Default Status
        ]);

        return response()->json([
            'success' => true,
            'data' => $order
        ], 201);
    }

    // Track Order with Status
    public function trackOrder($orderId)
    {
        $order = OrderConfirmation::where('order_id', $orderId)->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $order->order_id,
                'order_date' => $order->order_date,
                'expected_delivery_date' => $order->expected_delivery_date,
                'confirmation_message' => $order->confirmation_message,
                'track_order_url' => $order->track_order_url,
                'status' => $order->status
            ]
        ], 200);
    }

    // Update Order Status
    public function updateStatus(Request $request, $orderId)
    {
        $request->validate([
            'status' => 'required|in:Pending,Processing,Shipped,Delivered,Cancelled'
        ]);

        $order = OrderConfirmation::where('order_id', $orderId)->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $order->update([
            'status' => $request->status
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Order status updated successfully',
            'data' => $order
        ], 200);
    }
}
