<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use App\Models\User;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');  // Ensure user is authenticated
    }

    // Get notifications for a specific beautician
    public function getNotifications($beauticianId)
    {
        // Ensure beautician exists
        $beautician = User::find($beauticianId);
        if (!$beautician) {
            return response()->json([
                'message' => 'Beautician not found'
            ], 404);
        }

        // Fetch notifications for the beautician
        $notifications = Notification::where('beautician_id', $beauticianId)
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'notifications' => $notifications,
        ], 200);
    }

    // Mark a notification as read
    public function markAsRead($id)
    {
        $notification = Notification::find($id);

        if (!$notification) {
            return response()->json([
                'message' => 'Notification not found'
            ], 404);
        }

        $notification->update(['is_read' => true]);

        return response()->json([
            'message' => 'Notification marked as read'
        ], 200);
    }

    // Send a new notification (when review is added or order request is made)
    public function createNotification(Request $request)
    {
        $request->validate([
            'beautician_id' => 'required|exists:users,id',
            'title' => 'required|string',
            'message' => 'required|string',
            'type' => 'required|in:order_request,review,appointment',
        ]);

        $notification = Notification::create([
            'beautician_id' => $request->beautician_id,
            'title' => $request->title,
            'message' => $request->message,
            'type' => $request->type,
        ]);

        return response()->json([
            'notification' => $notification,
            'message' => 'Notification created successfully'
        ], 201);
    }
}
