<?php
namespace App\Http\Controllers;

use App\Models\Expertise;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class ExpertiseController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    // Add Expertise
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'expertise_name' => 'required|string|max:255',
            'expertise_description' => 'nullable|string',
            'expertise_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $imagePath = null;
        if ($request->hasFile('expertise_image')) {
            $imagePath = $request->file('expertise_image')->store('expertises', 'public');
        }

        $expertise = Expertise::create([
            'beautician_id' => $user->id,
            'expertise_name' => $request->expertise_name,
            'expertise_description' => $request->expertise_description,
            'expertise_image' => $imagePath
        ]);

        return response()->json(['message' => 'Expertise added successfully', 'expertise' => $expertise], 201);
    }

    // View All Expertise
    public function index()
    {
        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $expertises = Expertise::where('beautician_id', $user->id)->get();
        return response()->json(['expertises' => $expertises], 200);
    }

    // Update Expertise
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'expertise_name' => 'required|string|max:255',
            'expertise_description' => 'nullable|string',
            'expertise_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $user = Auth::user();
        $expertise = Expertise::where('beautician_id', $user->id)->where('id', $id)->first();

        if (!$expertise) {
            return response()->json(['message' => 'Expertise not found'], 404);
        }

        if ($request->hasFile('expertise_image')) {
            if ($expertise->expertise_image) {
                Storage::disk('public')->delete($expertise->expertise_image);
            }
            $expertise->expertise_image = $request->file('expertise_image')->store('expertises', 'public');
        }

        $expertise->update([
            'expertise_name' => $request->expertise_name,
            'expertise_description' => $request->expertise_description
        ]);

        return response()->json(['message' => 'Expertise updated successfully', 'expertise' => $expertise], 200);
    }

    // Delete Expertise
    public function destroy($id)
    {
        $user = Auth::user();
        $expertise = Expertise::where('beautician_id', $user->id)->where('id', $id)->first();

        if (!$expertise) {
            return response()->json(['message' => 'Expertise not found'], 404);
        }

        if ($expertise->expertise_image) {
            Storage::disk('public')->delete($expertise->expertise_image);
        }

        $expertise->delete();
        return response()->json(['message' => 'Expertise deleted successfully'], 200);
    }
}
