<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use App\Models\Experience;
use Illuminate\Support\Facades\Storage;

class ExperienceController extends Controller
{
    public function index(): JsonResponse
    {
        $user = Auth::user();
        $experiences = Experience::where('beautician_id', $user->id)->get();

        return response()->json([
            'status' => true,
            'message' => 'Experiences retrieved successfully',
            'data' => $experiences
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'job_title' => 'required|string|max:255',
            'company_name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'experience_description' => 'required|string',
            'certificate_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $user = Auth::user();

        $experience = new Experience();
        $experience->beautician_id = $user->id;
        $experience->job_title = $request->job_title;
        $experience->company_name = $request->company_name;
        $experience->start_date = $request->start_date;
        $experience->end_date = $request->end_date;
        $experience->experience_description = $request->experience_description;

        if ($request->hasFile('certificate_image')) {
            $path = $request->file('certificate_image')->store('certificates', 'public');
            $experience->certificate_image = $path;
        }

        $experience->save();

        return response()->json([
            'status' => true,
            'message' => 'Experience added successfully',
            'data' => $experience
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        $experience = Experience::where('beautician_id', $user->id)->where('id', $id)->first();

        if (!$experience) {
            return response()->json([
                'status' => false,
                'message' => 'Experience not found'
            ], 404);
        }

        $request->validate([
            'job_title' => 'sometimes|string|max:255',
            'company_name' => 'sometimes|string|max:255',
            'start_date' => 'sometimes|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'experience_description' => 'sometimes|string',
            'certificate_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $experience->update($request->only([
            'job_title', 'company_name', 'start_date', 'end_date', 'experience_description'
        ]));

        if ($request->hasFile('certificate_image')) {
            Storage::disk('public')->delete($experience->certificate_image);
            $path = $request->file('certificate_image')->store('certificates', 'public');
            $experience->certificate_image = $path;
            $experience->save();
        }

        return response()->json([
            'status' => true,
            'message' => 'Experience updated successfully',
            'data' => $experience
        ]);
    }

    public function destroy($id): JsonResponse
    {
        $user = Auth::user();
        $experience = Experience::where('beautician_id', $user->id)->where('id', $id)->first();

        if (!$experience) {
            return response()->json([
                'status' => false,
                'message' => 'Experience not found'
            ], 404);
        }

        Storage::disk('public')->delete($experience->certificate_image);
        $experience->delete();

        return response()->json([
            'status' => true,
            'message' => 'Experience deleted successfully'
        ]);
    }
}
