<?php

namespace App\Http\Controllers;

use App\Models\CartItem;
use App\Models\Product;
use Illuminate\Http\Request;

class CartController extends Controller
{
    // Show favorite products in the cart
    public function index(Request $request)
    {
        $cartItems = CartItem::with('product')
            ->where('user_id', $request->user_id)
            ->where('is_favorite', true) // Show only favorite products
            ->get();

        return response()->json([
            'cart_items' => $cartItems->map(function ($item) {
                return [
                    'id' => $item->id,
                    'product' => [
                        'id' => $item->product->id,
                        'name' => $item->product->name,
                        'image' => $item->product->image,
                        'rating' => $item->product->rating,
                        'reviews' => $item->product->reviews,
                        'price' => $item->product->price,
                        'like' => $item->is_favorite, // Dynamically check if liked
                        'share_url' => url("/product/" . $item->product->id), // Share link
                        'icon_url' => url("/product/details/" . $item->product->id), // Product details icon
                    ],
                    'quantity' => $item->quantity,
                    'delete_option' => true, // Allow deletion
                ];
            })
        ]);
    }

    // Add product to cart
    public function addToCart(Request $request)
    {
        try {
            $cartItem = CartItem::create([
                'user_id' => $request->user_id,
                'product_id' => $request->product_id,
                'quantity' => $request->quantity ?? 1,
                'is_favorite' => $request->is_favorite ?? false, // Favorite status
            ]);

            return response()->json([
                'message' => 'Product added to cart',
                'data' => $cartItem,
                'status' => 200
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Something went wrong',
                'error' => $e->getMessage(),
                'status' => 500
            ], 500);
        }
    }

    // Remove product from cart
    public function removeFromCart(Request $request)
    {
        try {
            $cartItem = CartItem::where('user_id', $request->user_id)
                ->where('product_id', $request->product_id)
                ->first();

            if (!$cartItem) {
                return response()->json([
                    'message' => 'Item not found in cart',
                    'status' => 404
                ], 404);
            }

            $cartItem->delete();

            return response()->json([
                'message' => 'Item removed successfully',
                'status' => 200
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Something went wrong',
                'error' => $e->getMessage(),
                'status' => 500
            ], 500);
        }
    }

    // Update quantity of a product in the cart
    public function updateCart(Request $request)
    {
        try {
            $request->validate([
                'user_id' => 'required|integer',
                'product_id' => 'required|integer',
                'quantity' => 'required|integer|min:1',
            ]);

            $cartItem = CartItem::where('user_id', $request->user_id)
                                ->where('product_id', $request->product_id)
                                ->first();

            if (!$cartItem) {
                return response()->json([
                    'message' => 'Cart item not found',
                    'status' => 404
                ], 404);
            }

            $cartItem->update([
                'quantity' => $request->quantity,
                'is_favorite' => $request->is_favorite ?? $cartItem->is_favorite, // Allow favorite update
            ]);

            return response()->json([
                'message' => 'Cart updated successfully',
                'data' => $cartItem,
                'status' => 200
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Something went wrong',
                'error' => $e->getMessage(),
                'status' => 500
            ], 500);
        }
    }
}
