<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Skill;

class BeauticianSkillController extends Controller
{
    // Ensure only authenticated beauticians can manage skills
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    // 1️⃣ Add Skill
    public function addSkill(Request $request)
    {
        $request->validate([
            'skill_name' => 'required|string|max:255',
            'skill_picture' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Access denied! Only beauticians can add skills.'], 403);
        }

        $skill_picture_path = null;
        if ($request->hasFile('skill_picture')) {
            $skill_picture_path = $request->file('skill_picture')->store('skills', 'public');
        }

        $skill = Skill::create([
            'beautician_id' => $user->id,
            'skill_name' => $request->skill_name,
            'skill_picture' => $skill_picture_path,
        ]);

        return response()->json(['message' => 'Skill added successfully!', 'skill' => $skill], 201);
    }

    // 2️⃣ View All Skills
    public function viewSkills()
    {
        $user = Auth::user();
        if ($user->role !== 'beautician') {
            return response()->json(['message' => 'Access denied!'], 403);
        }

        $skills = Skill::where('beautician_id', $user->id)->get();
        return response()->json(['skills' => $skills]);
    }

    // 3️⃣ Update Skill
    public function updateSkill(Request $request, $id)
    {

        $request->validate([
            'skill_name' => 'required|string|max:255',
            'skill_picture' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $user = Auth::user();
        $skill = Skill::where('beautician_id', $user->id)->where('id', $id)->first();

        if (!$skill) {
            return response()->json(['message' => 'Skill not found or unauthorized access.'], 404);
        }

        $skill->skill_name = $request->skill_name;

        if ($request->hasFile('skill_picture')) {
            $skill_picture_path = $request->file('skill_picture')->store('skills', 'public');
            $skill->skill_picture = $skill_picture_path;
        }

        $skill->save();

        return response()->json(['message' => 'Skill updated successfully!', 'skill' => $skill]);
    }

    // 4️⃣ Delete Skill
    public function deleteSkill($id)
    {
        $user = Auth::user();
        $skill = Skill::where('beautician_id', $user->id)->where('id', $id)->first();

        if (!$skill) {
            return response()->json(['message' => 'Skill not found or unauthorized access.'], 404);
        }

        $skill->delete();
        return response()->json(['message' => 'Skill deleted successfully!']);
    }
}
