<?php

namespace App\Http\Controllers;

use App\Models\BeauticianPortfolio;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class BeauticianPortfolioController extends Controller
{
    // ✅ Get Portfolio
    public function index($beautician_id)
    {
        $portfolio = BeauticianPortfolio::where('beautician_id', $beautician_id)->first();

        if (!$portfolio) {
            return response()->json(['message' => 'Portfolio not found'], 404);
        }

        return response()->json($portfolio, 200);
    }

    // ✅ Add Portfolio
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'beautician_id' => 'required|exists:users,id',
            'name' => 'nullable|string|max:255',
            'portfolio_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'youtube_link' => 'nullable|url',
            'facebook_link' => 'nullable|url',
            'instagram_link' => 'nullable|url',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Handle Image Upload
        $imagePath = null;
        if ($request->hasFile('portfolio_image')) {
            $imagePath = $request->file('portfolio_image')->store('portfolio_images', 'public');
        }

        $portfolio = BeauticianPortfolio::create([
            'beautician_id' => $request->beautician_id,
            'name' => $request->name,
            'portfolio_image' => $imagePath,
            'youtube_link' => $request->youtube_link,
            'facebook_link' => $request->facebook_link,
            'instagram_link' => $request->instagram_link,
        ]);

        return response()->json(['portfolio' => $portfolio, 'message' => 'Portfolio created successfully'], 201);
    }

    // ✅ Update Portfolio
    public function update(Request $request, $beautician_id)
    {
        $portfolio = BeauticianPortfolio::where('beautician_id', $beautician_id)->first();

        if (!$portfolio) {
            return response()->json(['message' => 'Portfolio not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'nullable|string|max:255',
            'portfolio_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'youtube_link' => 'nullable|url',
            'facebook_link' => 'nullable|url',
            'instagram_link' => 'nullable|url',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Handle Image Update
        if ($request->hasFile('portfolio_image')) {
            if ($portfolio->portfolio_image) {
                Storage::disk('public')->delete($portfolio->portfolio_image);
            }
            $portfolio->portfolio_image = $request->file('portfolio_image')->store('portfolio_images', 'public');
        }

        $portfolio->update($request->except(['portfolio_image']));

        return response()->json(['portfolio' => $portfolio, 'message' => 'Portfolio updated successfully'], 200);
    }

    // ✅ Delete Portfolio
    public function destroy($beautician_id)
    {
        $portfolio = BeauticianPortfolio::where('beautician_id', $beautician_id)->first();

        if (!$portfolio) {
            return response()->json(['message' => 'Portfolio not found'], 404);
        }

        if ($portfolio->portfolio_image) {
            Storage::disk('public')->delete($portfolio->portfolio_image);
        }

        $portfolio->delete();

        return response()->json(['message' => 'Portfolio deleted successfully'], 200);
    }
}
