<?php
namespace App\Http\Controllers;

use App\Models\BeauticianClientPicture;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class BeauticianClientPictureController extends Controller
{
    // ✅ View all pictures for a beautician
    public function index($beautician_id)
    {
        $pictures = BeauticianClientPicture::where('beautician_id', $beautician_id)
                    ->orderByDesc('created_at')
                    ->select('id', 'title', 'image_path', 'created_at')
                    ->get();

        return response()->json(['pictures' => $pictures], 200);
    }

    // ✅ Add picture
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'beautician_id' => 'required|exists:users,id',
            'title' => 'nullable|string|max:255',
            'image' => 'required|image|mimes:jpeg,jpg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $path = $request->file('image')->store('client_pictures', 'public');

        $picture = BeauticianClientPicture::create([
            'beautician_id' => $request->beautician_id,
            'title' => $request->title,
            'image_path' => $path,
        ]);

        return response()->json(['picture' => $picture, 'message' => 'Picture uploaded successfully'], 201);
    }

    // ✅ Update picture title or image
    public function update(Request $request, $id)
    {
        $picture = BeauticianClientPicture::find($id);

        if (!$picture) {
            return response()->json(['message' => 'Picture not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,jpg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        if ($request->hasFile('image')) {
            Storage::disk('public')->delete($picture->image_path);
            $picture->image_path = $request->file('image')->store('client_pictures', 'public');
        }

        if ($request->title) {
            $picture->title = $request->title;
        }

        $picture->save();

        return response()->json(['picture' => $picture, 'message' => 'Picture updated successfully'], 200);
    }

    // ✅ Delete picture
    public function destroy($id)
    {
        $picture = BeauticianClientPicture::find($id);

        if (!$picture) {
            return response()->json(['message' => 'Picture not found'], 404);
        }

        Storage::disk('public')->delete($picture->image_path);
        $picture->delete();

        return response()->json(['message' => 'Picture deleted successfully'], 200);
    }
}
