<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    // User Registration
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'phone' => 'required|string|max:15',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,

        ]);

        return response()->json([
            'message' => 'User registered successfully',
            'user' => $user
        ], 201);
    }

    // User Login
    public function login(Request $request)
    {


        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'remember_me' => 'boolean'
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }

        // Generate token
        $token = $user->createToken('auth_token')->plainTextToken;

        // Store remember_me status in a session or database
        if ($request->remember_me) {
            $user->tokens()->update(['expires_at' => now()->addWeeks(2)]);
        } else {
            $user->tokens()->update(['expires_at' => now()->addHours(2)]);
        }

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }

        if (!$user->is_verified) {
            return response()->json(['message' => 'Your account is not verified yet. Please wait for admin approval.'], 403);
        }

        return response()->json([
            'message' => 'Login successful',
            'access_token' => $token,
            'token_type' => 'Bearer',
            'user' => $user
        ]);
    }


    // User Logout
    public function logout(Request $request)
    {
        $user = $request->user();
        $token = $user->tokens()->where('id', $user->currentAccessToken()->id)->first();

        if ($token) {
            $token->delete(); // Delete current token
        }

        return response()->json([
            'message' => 'Successfully logged out'
        ]);
    }

    // Get Authenticated User
    public function profile(Request $request)
    {
        return response()->json($request->user());
    }

    public function registerAdmin(Request $request)
    {
        // Validate the request
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6'
        ]);

        // Create the admin user
        $admin = new User();
        $admin->name = $request->name;
        $admin->email = $request->email;
        $admin->password = Hash::make($request->password);
        $admin->role = 'admin'; // Ensure role is set to 'admin'
        $admin->save();

        return response()->json([
            'status' => true,
            'message' => 'Admin user created successfully',
            'user' => $admin
        ], 201);
    }

    public function adminLogin(Request $request)
{
    $request->validate([
        'email' => 'required|email',
        'password' => 'required'
    ]);

    // Find user by email
    $user = User::where('email', $request->email)->where('role', 'admin')->first();

    if (!$user || !Hash::check($request->password, $user->password)) {
        return response()->json(['message' => 'Invalid admin credentials'], 401);
    }

    $token = $user->createToken('admin_auth_token')->plainTextToken;

    return response()->json([
        'status' => true,
        'message' => 'Admin login successful',
        'token' => $token
    ]);
}


public function registerBeautician(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|string|email|max:255|unique:users',
        'password' => 'required|string|min:6',
        'phone' => 'required|string|max:15',

    ]);

    $beautician = new User();
    $beautician->name = $request->name;
    $beautician->email = $request->email;
    $beautician->password = Hash::make($request->password);
    $beautician->phone = $request->phone;
    $beautician->role = 'beautician'; // Ensure the role is set
    $beautician->save();

    return response()->json([
        'status' => true,
        'message' => 'Beautician registered successfully',
        'user' => $beautician
    ], 201);
}


public function loginBeautician(Request $request)
{
    $request->validate([
        'email' => 'required|string|email',
        'password' => 'required|string'
    ]);

    $user = User::where('email', $request->email)->first();

    if (!$user || !Hash::check($request->password, $user->password)) {
        return response()->json(['message' => 'Invalid beautician credentials'], 401);
    }

    if ($user->role !== 'beautician') {
        return response()->json(['message' => 'Unauthorized: Not a beautician'], 403);
    }

    $token = $user->createToken('beautician-token')->plainTextToken;

    if (!$user || !Hash::check($request->password, $user->password)) {
        return response()->json(['message' => 'Invalid credentials'], 401);
    }

    if (!$user->is_verified) {
        return response()->json(['message' => 'Your account is not verified yet. Please wait for admin approval.'], 403);
    }


    return response()->json([
        'status' => true,
        'message' => 'Beautician logged in successfully',
        'token' => $token,
        'user' => $user
    ]);
}


public function verifyUser(Request $request, $id)
{
    $admin = $request->user();

    if ($admin->role !== 'admin') {
        return response()->json(['message' => 'Unauthorized'], 403);
    }

    $user = User::find($id);

    if (!$user) {
        return response()->json(['message' => 'User not found'], 404);
    }

    $user->is_verified = true;
    $user->save();

    return response()->json([
        'status' => true,
        'message' => 'User has been verified successfully.',
        'user' => $user
    ]);
}


public function unverifiedUsers(Request $request)
{
    $admin = $request->user();

    if ($admin->role !== 'admin') {
        return response()->json(['message' => 'Unauthorized'], 403);
    }

    $perPage = $request->get('per_page', 10);
    $sortOrder = $request->get('sort', 'desc'); // asc or desc
    $role = $request->get('role'); // e.g. beautician, salon, etc.
    $searchKeyword = $request->get('search'); // For name or email search

    $query = User::where('is_verified', false);

    if ($role) {
        $query->where('role', $role);
    }

    // If there's a search keyword, apply the filter
    if ($searchKeyword) {
        $query->where(function ($q) use ($searchKeyword) {
            $q->where('name', 'like', "%$searchKeyword%")
              ->orWhere('email', 'like', "%$searchKeyword%");
        });
    }

    $users = $query->orderBy('created_at', $sortOrder)->paginate($perPage);

    return response()->json([
        'status' => true,
        'message' => 'List of unverified users' . ($role ? " with role: $role" : '') . ($searchKeyword ? " matching '$searchKeyword'" : ''),
        'users' => $users
    ]);
}





}
