<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Order;
use App\Models\Appointment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Response;
use Barryvdh\DomPDF\Facade\Pdf;


class AdminReportController extends Controller
{
    public function overview()
    {
        // User Stats
        $totalUsers = User::count();
        $newUsers = User::where('created_at', '>=', Carbon::now()->subDays(30))->count();
        $activeUsers = User::whereHas('orders')->orWhereHas('appointments')->distinct()->count('id');

        // Beautician Stats (assuming role = beautician)
        $totalBeauticians = User::where('role', 'beautician')->count();
        $beauticianAppointments = Appointment::whereHas('user', function ($q) {
            $q->where('role', 'beautician');
        })->count();
        $beauticianRatings = Appointment::whereNotNull('rating')->avg('rating');

        // Earnings
        $orderEarnings = Order::where('status', 'to_pay')->sum(DB::raw('price * quantity'));
        // If appointments are paid, we can add this too (optional)
        // $appointmentEarnings = Appointment::sum('price');

        // System Performance
        $totalAppointments = Appointment::count();
        $statusCounts = Appointment::select('status', DB::raw('count(*) as total'))
                                   ->groupBy('status')
                                   ->pluck('total', 'status');

        return response()->json([
            'status' => true,
            'message' => 'Admin Report Overview',
            'data' => [
                'users' => [
                    'total' => $totalUsers,
                    'new_last_30_days' => $newUsers,
                    'active_users' => $activeUsers,
                ],
                'beauticians' => [
                    'total' => $totalBeauticians,
                    'appointments_handled' => $beauticianAppointments,
                    'average_rating' => round($beauticianRatings, 1),
                ],
                'earnings' => [
                    'from_orders' => $orderEarnings,
                    // 'from_appointments' => $appointmentEarnings, // if applicable
                ],
                'system_performance' => [
                    'total_appointments' => $totalAppointments,
                    'appointments_by_status' => $statusCounts,
                ],
            ]
        ]);
    }

    public function exportReport(Request $request, $type = 'pdf')
{
    // Collect same report data
    $data = $this->getReportData();

    if ($type === 'pdf') {
        $pdf = Pdf::loadView('admin.report_pdf', $data);
        return $pdf->download('admin_report.pdf');
    }

    if ($type === 'csv') {
        $csvFileName = 'admin_report.csv';
        $headers = [
            'Content-type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=$csvFileName",
        ];

        $callback = function () use ($data) {
            $handle = fopen('php://output', 'w');

            fputcsv($handle, ['Section', 'Label', 'Value']);

            foreach ($data as $section => $values) {
                foreach ($values as $key => $val) {
                    fputcsv($handle, [ucfirst($section), ucfirst(str_replace('_', ' ', $key)), $val]);
                }
            }

            fclose($handle);
        };

        return Response::stream($callback, 200, $headers);
    }

    return response()->json(['error' => 'Invalid export type'], 400);
}

// Helper method to reuse data logic
private function getReportData()
{
    $totalUsers = User::count();
    $newUsers = User::where('created_at', '>=', now()->subDays(30))->count();
    $activeUsers = User::whereHas('orders')->orWhereHas('appointments')->distinct()->count('id');

    $totalBeauticians = User::where('role', 'beautician')->count();
    $beauticianAppointments = Appointment::whereHas('user', fn($q) => $q->where('role', 'beautician'))->count();
    $beauticianRatings = Appointment::whereNotNull('rating')->avg('rating');

    $orderEarnings = Order::where('status', 'to_pay')->sum(DB::raw('price * quantity'));
    $totalAppointments = Appointment::count();
    $statusCounts = Appointment::select('status', DB::raw('count(*) as total'))->groupBy('status')->pluck('total', 'status');

    return [
        'users' => [
            'total' => $totalUsers,
            'new_last_30_days' => $newUsers,
            'active_users' => $activeUsers,
        ],
        'beauticians' => [
            'total' => $totalBeauticians,
            'appointments_handled' => $beauticianAppointments,
            'average_rating' => round($beauticianRatings, 1),
        ],
        'earnings' => [
            'from_orders' => $orderEarnings,
        ],
        'system_performance' => [
            'total_appointments' => $totalAppointments,
            'appointments_by_status' => json_encode($statusCounts),
        ],
    ];
}
}
